<?php
// content.php — Crypto News: single article page with slug routing, share buttons, JSON-LD

/////////////////////////////
// 0) OPTIONAL DEBUG MODE  //
/////////////////////////////
// Add &debug=1 to the URL to see PHP errors and debugging headers
$DEBUG = isset($_GET['debug']);
if ($DEBUG) {
  ini_set('display_errors', 1);
  ini_set('display_startup_errors', 1);
  error_reporting(E_ALL);
  header('X-Debug-File', __FILE__);
}

/////////////////////////////
// 1) LOAD DATA SAFELY     //
/////////////////////////////
$dataPath = __DIR__ . '/news-data.php';
if (!file_exists($dataPath)) {
  http_response_code(500);
  echo "news-data.php not found at: " . htmlspecialchars($dataPath);
  exit;
}
require_once $dataPath;

if (!isset($ARTICLES) || !is_array($ARTICLES)) {
  http_response_code(500);
  echo "\$ARTICLES not defined or not an array in news-data.php";
  exit;
}

/////////////////////////////
// 2) HELPERS              //
/////////////////////////////
function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }
function base_url(){
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
  return $scheme . "://" . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
}
function slugify($t){
  $t = function_exists('mb_strtolower') ? mb_strtolower(trim($t ?? ''), 'UTF-8') : strtolower(trim($t ?? ''));
  // English-only slugs (a–z, 0–9, hyphen)
  $t = preg_replace('~[^\pL0-9]+~u', '-', $t);
  $t = trim($t, '-');
  $t = preg_replace('~[^-a-z0-9]+~u', '', $t);
  return $t ?: 'news';
}

/////////////////////////////
// 3) BUILD SLUG → ARTICLE //
/////////////////////////////
$map = [];
$seen = [];
foreach ($ARTICLES as $a) {
  $slug = $a['slug'] ?? slugify($a['title'] ?? '');
  if (isset($seen[$slug])) {
    $slug .= '-'.substr(md5(($a['title'] ?? '').($a['datePublished'] ?? '')),0,6);
  }
  $seen[$slug] = true;
  $a['slug'] = $slug;
  $map[$slug] = $a;
}

/////////////////////////////
// 4) RESOLVE CURRENT SLUG //
/////////////////////////////
$slug = null;

// 4a) Query param version (?slug=…)
if (isset($_GET['slug']) && $_GET['slug'] !== '') {
  $slug = rawurldecode($_GET['slug']);
}

// 4b) Pretty URL fallback: /crypto-news/<slug>
if (!$slug) {
  $uriPath = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
  $uriPath = rtrim($uriPath, '/');
  $last = basename($uriPath);
  if ($last === 'index.php' || $last === 'content.php' || $last === '') {
    $slug = null;
  } else {
    $slug = rawurldecode($last);
  }
}

if ($DEBUG) {
  header('X-Debug-QueryString: '.($_SERVER['QUERY_STRING'] ?? ''));
  header('X-Debug-Resolved-Slug: '.($slug ?? 'NULL'));
}

// Normalize slug (allow a–z, 0–9, hyphen)
if ($slug !== null) {
  $slug = trim($slug);
  $slug = preg_replace('~[^-a-z0-9]+~u', '', $slug);
  if ($slug === '') $slug = null;
}

/////////////////////////////
// 5) PICK ARTICLE OR 404  //
/////////////////////////////
$article = ($slug && isset($map[$slug])) ? $map[$slug] : null;

if (!$article) {
  http_response_code(404);
  ?>
  <!doctype html>
  <meta charset="utf-8">
  <title>News Not Found (404)</title>
  <style>
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Arial,sans-serif;background:#0b0c10;color:#f5f7fb;margin:0;display:grid;place-items:center;height:100vh}
    .box{max-width:720px;padding:24px;border:1px solid #1b1d25;background:#12131a;border-radius:16px}
    a{color:#ff6b3d;text-decoration:none}
  </style>
  <div class="box">
    <h1>404 — News Not Found</h1>
    <p>The page you are looking for is not available.</p>
    <p><a href="index.php">Go to Home Page</a></p>
  </div>
  <?php
  exit;
}

/////////////////////////////
// 6) PREP SEO & CONTENT   //
/////////////////////////////
$images = [];
if (!empty($article['images']) && is_array($article['images'])) {
  $images = array_values(array_filter($article['images']));
} elseif (!empty($article['image'])) {
  $images = [$article['image']];
} else {
  $images = ['https://via.placeholder.com/1200x630?text=News'];
}
$firstImg   = $images[0];
$canonical  = base_url().'/'.rawurlencode($article['slug']);
$shareURL   = $canonical;
$published  = $article['datePublished'] ?? date('Y-m-d');
$modified   = $article['dateModified']  ?? $published;
$author     = $article['author'] ?? '';
$category   = $article['category'] ?? '';
$publisherName = "Crypto News";
$publisherLogo = "https://via.placeholder.com/600x60?text=Crypto+News+Logo";
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title><?php echo h($article['title']); ?> — Crypto News</title>
  <meta name="description" content="<?php echo h($article['short'] ?? $article['long'] ?? ''); ?>" />
  <link rel="canonical" href="<?php echo h($canonical); ?>" />
  <link rel="stylesheet" href="style.css" />

  <!-- Open Graph / Twitter -->
  <meta property="og:site_name" content="Crypto News" />
  <meta property="og:title" content="<?php echo h($article['title']); ?>" />
  <meta property="og:description" content="<?php echo h($article['short'] ?? $article['long'] ?? ''); ?>" />
  <meta property="og:image" content="<?php echo h($firstImg); ?>" />
  <meta property="og:url" content="<?php echo h($shareURL); ?>" />
  <meta property="og:type" content="article" />
  <meta name="twitter:card" content="summary_large_image" />

  <!-- NewsArticle JSON-LD -->
  <script type="application/ld+json">
  {
    "@context": "https://schema.org",
    "@type": "NewsArticle",
    "headline": <?php echo json_encode($article['title']); ?>,
    "image": <?php echo json_encode($images); ?>,
    "datePublished": <?php echo json_encode($published); ?>,
    "dateModified": <?php echo json_encode($modified); ?>,
    "author": { "@type": "Person", "name": <?php echo json_encode($author); ?> },
    "publisher": {
      "@type": "Organization",
      "name": <?php echo json_encode($publisherName); ?>,
      "logo": { "@type": "ImageObject", "url": <?php echo json_encode($publisherLogo); ?> }
    },
    "description": <?php echo json_encode($article['short'] ?? $article['long'] ?? ''); ?>,
    "articleSection": <?php echo json_encode($category); ?>,
    "url": <?php echo json_encode($shareURL); ?>
  }
  </script>

  <!-- Page-only CSS (auto-fit images in slider + body) -->
  <style>
    .slider{position:relative;background:#0f1118;border:1px solid var(--border);border-radius:16px;overflow:hidden;margin:12px 0}
    .slides img{width:100%;max-height:480px;object-fit:contain;display:none;background:#000}
    .slides img.active{display:block}
    .nav{position:absolute;top:50%;left:0;right:0;display:flex;justify-content:space-between;transform:translateY(-50%);padding:0 10px;pointer-events:none}
    .nav button{pointer-events:auto;background:rgba(0,0,0,.45);border:1px solid var(--border);color:#fff;border-radius:999px;padding:8px 12px;cursor:pointer}
    @media (max-width:720px){.slides img{max-height:320px}}
    .detail-body img{max-width:100%;height:auto;object-fit:contain;display:block;margin:10px auto;background:#000;border-radius:8px}
    .share{display:flex;gap:8px;flex-wrap:wrap;margin-top:14px}
  </style>
  
 
  
</head>
<body>
<header class="site-header">
  <div class="brand"><span class="fire">📰</span><a class="title" href="index.php">Crypto News</a></div>
  <nav class="main-nav">
    <a href="index.php">Home</a>
  </nav>
</header>

<main class="container">
  <article class="detail">
    <!-- Slider -->
    <div class="slider" id="newsSlider" aria-label="Article images">
      <div class="slides">
        <?php foreach ($images as $i => $src): ?>
          <img src="<?php echo h($src); ?>" alt="<?php echo h($article['title']); ?>"
               class="<?php echo $i===0 ? 'active' : ''; ?>"
               loading="<?php echo $i===0 ? 'eager' : 'lazy'; ?>"
               onerror="this.onerror=null;this.src='https://via.placeholder.com/1200x600?text=News';" />
        <?php endforeach; ?>
      </div>
      <?php if (count($images) > 1): ?>
        <div class="nav" aria-hidden="false">
          <button type="button" data-dir="-1" aria-label="Previous">«</button>
          <button type="button" data-dir="1" aria-label="Next">»</button>
        </div>
      <?php endif; ?>
    </div>

    <div class="detail-body">
      <h1 class="detail-title"><?php echo h($article['title']); ?></h1>
      <div class="meta big">
        <span>✍️ <?php echo h($author); ?></span>
        <span>🗂️ <?php echo h($category); ?></span>
        <span>📅 Published: <?php echo date('F j, Y', strtotime($published)); ?></span>
        <?php if ($modified && $modified !== $published): ?>
          <span>♻️ Updated: <?php echo date('F j, Y', strtotime($modified)); ?></span>
        <?php endif; ?>
      </div>

      <p class="lead"><?php echo h($article['short'] ?? ''); ?></p>
      <p><?php echo nl2br(h($article['long'] ?? '')); ?></p>

      <div class="share">
        <a class="btn ghost" target="_blank" rel="noopener noreferrer"
           href="https://www.facebook.com/sharer/sharer.php?u=<?php echo rawurlencode($shareURL); ?>">Facebook</a>
        <a class="btn ghost" target="_blank" rel="noopener noreferrer"
           href="https://api.whatsapp.com/send?text=<?php echo rawurlencode(($article['title'] ?? 'News') . ' ' . $shareURL); ?>">WhatsApp</a>
        <a class="btn ghost" target="_blank" rel="noopener noreferrer"
           href="https://www.linkedin.com/sharing/share-offsite/?url=<?php echo rawurlencode($shareURL); ?>">LinkedIn</a>
        <button class="btn ghost" id="copyLinkBtn">Copy Link</button>
      </div>
    </div>
  </article>
</main>

<footer class="site-footer"><div>Crypto News © <?php echo date('Y'); ?> • All rights reserved</div></footer>

<script>
(function(){
  // --- Slider: robust init + autoplay ---
  const INTERVAL = 5000; // ms

  function initSlider(){
    const wrap = document.getElementById('newsSlider');
    if (!wrap) return;

    const imgs = Array.from(wrap.querySelectorAll('.slides img'));
    if (!imgs.length) return;

    let i = imgs.findIndex(im => im.classList.contains('active'));
    if (i < 0) { i = 0; imgs[0].classList.add('active'); }

    const prev = wrap.querySelector('[data-dir="-1"]');
    const next = wrap.querySelector('[data-dir="1"]');

    let timer = null, paused = false;

    function show(n){
      i = (n + imgs.length) % imgs.length;
      imgs.forEach((im, idx) => im.classList.toggle('active', idx === i));
    }
    function schedule(){
      clearTimeout(timer);
      if (!paused && imgs.length > 1) {
        timer = setTimeout(() => { show(i + 1); schedule(); }, INTERVAL);
      }
    }
    function start(){ paused = false; schedule(); }
    function stop(){ paused = true; clearTimeout(timer); }

    prev && prev.addEventListener('click', () => { stop(); show(i - 1); start(); });
    next && next.addEventListener('click', () => { stop(); show(i + 1); start(); });

    // Touch swipe
    let sx = 0;
    wrap.addEventListener('touchstart', e => { sx = e.changedTouches[0].clientX; stop(); }, {passive:true});
    wrap.addEventListener('touchend',   e => {
      const dx = e.changedTouches[0].clientX - sx;
      if (Math.abs(dx) > 40) show(i + (dx < 0 ? 1 : -1));
      start();
    }, {passive:true});

    // Hover + tab visibility pause
    wrap.addEventListener('mouseenter', stop);
    wrap.addEventListener('mouseleave', start);
    document.addEventListener('visibilitychange', () => {
      if (document.hidden) stop(); else start();
    });

    show(i);
    start();
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initSlider, { once: true });
  } else {
    initSlider();
  }

  // Copy link
  const copyBtn = document.getElementById('copyLinkBtn');
  if (copyBtn){
    copyBtn.addEventListener('click', async () => {
      try{
        await navigator.clipboard.writeText(<?php echo json_encode($shareURL); ?>);
        copyBtn.textContent = 'Copied!';
        setTimeout(() => copyBtn.textContent = 'Copy Link', 1200);
      }catch(e){
        alert('Could not copy: ' + e);
      }
    });
  }
})();
</script>
</body>
</html>
