<?php
// index.php — Crypto News with Live Ticker
require_once __DIR__ . '/news-data.php';

function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }
function has_mb(){ return function_exists('mb_strtolower'); }
function lower($s){ return has_mb()? mb_strtolower($s,'UTF-8') : strtolower($s); }

function base_url(){
  $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
  $scheme = $https ? "https" : "http";
  return $scheme . "://" . $_SERVER['HTTP_HOST'] . rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
}
function base_path(){ return rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\'); }
function primary_image($a){
  if (!empty($a['images']) && is_array($a['images'])) return $a['images'][0];
  return $a['image'] ?? 'https://via.placeholder.com/1200x630?text=News';
}
function slugify($t){
  $t = lower(trim($t ?? ''));
  $t = preg_replace('~[^\pL0-9]+~u', '-', $t);
  $t = trim($t, '-');
  $t = preg_replace('~[^-a-z0-9\p{Devanagari}]+~u', '', $t);
  return $t ?: 'news';
}
function ensure_unique_slugs($articles){
  $seen = [];
  foreach ($articles as &$a){
    $base = $a['slug'] ?? slugify($a['title'] ?? '');
    $slug = $base;
    if (isset($seen[$slug])) $slug .= '-'.substr(md5(($a['title'] ?? '').($a['datePublished'] ?? '')),0,6);
    $seen[$slug] = true;
    $a['slug'] = $slug;
  }
  return $articles;
}
$ARTICLES = ensure_unique_slugs($ARTICLES);
usort($ARTICLES, function($a,$b){
  $da = strtotime($a['datePublished'] ?? '1970-01-01');
  $db = strtotime($b['datePublished'] ?? '1970-01-01');
  return $db <=> $da;
});

function pick_items($list, $n, $pred, &$taken = []) {
  $out = [];
  foreach ($list as $it) {
    $slug = $it['slug'] ?? '';
    if (isset($taken[$slug])) continue;
    if (!$pred($it)) continue;
    $out[] = $it;
    $taken[$slug] = true;
    if (count($out) >= $n) break;
  }
  return $out;
}
$now = time();
$RECENT_HOURS = 24*7;
function is_recent($a,$now,$h){ $ts=strtotime($a['datePublished']??'1970-01-01'); return ($now-$ts) <= ($h*3600); }

$taken=[];
$recent = pick_items($ARTICLES, 10, fn($a)=>is_recent($a,$now,$RECENT_HOURS), $taken);
if (!$recent) $recent = array_slice($ARTICLES, 0, min(10,count($ARTICLES)));

/* Search + pagination */
$tagQueryRaw = $_GET['tag'] ?? '';
$tagNeedles = array_filter(array_map('trim', preg_split('/[,\s]+/', $tagQueryRaw)));
$tagNeedles = array_map(fn($t)=> has_mb()? mb_strtolower($t,'UTF-8') : strtolower($t), $tagNeedles);
$filtered = array_filter($ARTICLES, function($a) use ($tagNeedles){
  if (!$tagNeedles) return true;
  $tags = array_map(fn($t)=> has_mb()? mb_strtolower($t,'UTF-8') : strtolower($t), $a['tags'] ?? []);
  foreach ($tagNeedles as $needle){ foreach ($tags as $tag){ if ($needle!=='' && strpos($tag,$needle)!==false) return true; } }
  return false;
});
usort($filtered, function($a,$b){
  $da = strtotime($a['datePublished'] ?? '1970-01-01');
  $db = strtotime($b['datePublished'] ?? '1970-01-01');
  return $db <=> $da;
});
$PER_PAGE=30; $total=count($filtered); $pages=max(1,(int)ceil($total/$PER_PAGE));
$pg = isset($_GET['pg']) ? max(1,(int)$_GET['pg']) : 1; if ($pg>$pages) $pg=$pages;
$offset = ($pg-1)*$PER_PAGE; $all_list = array_slice($filtered,$offset,$PER_PAGE);
$qs = function($page) use ($tagQueryRaw){
  $p=[]; if($tagQueryRaw!=='') $p['tag']=$tagQueryRaw; if($page>1) $p['pg']=$page;
  return $p ? ('?'.http_build_query($p)) : '';
};
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Crypto News 📰</title>
  <meta name="description" content="Crypto News — Breaking news and analysis." />
  <link rel="stylesheet" href="style.css?v=clean" />
</head>
<body>
<header class="site-header">
  <div class="brand"><span class="fire">₿</span><span class="title">Crypto News</span></div>
  <nav class="main-nav"><a href="<?php echo h(base_path().'/'); ?>">Home</a></nav>
</header>


<?php if (!empty($recent)): ?>
  <div class="ticker-wrap" aria-label="Breaking">
    <div class="ticker">
      <span class="ticker-label">Breaking</span>
      <div class="ticker-viewport">
        <div class="ticker-track">
          <?php foreach ($recent as $a): $url=base_path().'/'.rawurlencode($a['slug']); ?>
            <a href="<?php echo h($url); ?>"><?php echo h($a['title']); ?></a>
          <?php endforeach; ?>
          <span aria-hidden="true">
            <?php foreach ($recent as $a): $url=base_path().'/'.rawurlencode($a['slug']); ?>
              <a href="<?php echo h($url); ?>"><?php echo h($a['title']); ?></a>
            <?php endforeach; ?>
          </span>
        </div>
      </div>
    </div>
  </div>
<?php endif; ?>

<div class="ticker-wrap live-ticker-wrap">
  <div class="ticker live-ticker">
    <div class="ticker-viewport">
      <div class="ticker-track live-ticker-track" id="live-crypto-ticker"></div>
    </div>
  </div>
</div>

<main class="container">
  <section class="toolbar">
    <form class="search" method="get">
      <input type="text" name="tag" value="<?php echo h($tagQueryRaw); ?>" placeholder="Search by tag (e.g., bitcoin, ethereum, defi)" />
      <button type="submit">Search</button>
      <?php if ($tagQueryRaw!==''): ?><a href="index.php" class="btn ghost">Reset</a><?php endif; ?>
    </form>
  </section>

  <section>
    <h2 class="section-title">All News</h2>
    <?php if ($total===0): ?>
      <p class="muted">No stories found. Try different tags.</p>
    <?php else: ?>
      <div class="results-info">
        Showing <?php echo ($offset+1); ?>–<?php echo min($offset+count($all_list),$total); ?> of <?php echo $total; ?>
        <?php if ($tagQueryRaw!=='') echo ' — #'.h($tagQueryRaw); ?>
      </div>
    <?php endif; ?>

    <?php foreach ($all_list as $a): ?>
      <?php $slug=$a['slug']; $pretty=base_path().'/'.rawurlencode($slug); $abs=base_url().'/'.rawurlencode($slug); $thumb=primary_image($a); ?>
      <article class="card">
        <a class="thumb" href="<?php echo h($pretty); ?>">
          <img src="<?php echo h($thumb); ?>" alt="<?php echo h($a['title']); ?>" loading="lazy"
                  onerror="this.onerror=null;this.src='https://via.placeholder.com/1200x630?text=News';">
        </a>
        <div class="card-body">
          <h3 class="card-title"><a href="<?php echo h($pretty); ?>"><?php echo h($a['title']); ?></a></h3>
          <div class="meta">
            <span>✍️ <?php echo h($a['author'] ?? ''); ?></span>
            <span>🗂️ <?php echo h($a['category'] ?? ''); ?></span>
            <span>📅 <?php echo date('M j, Y', strtotime($a['datePublished'] ?? 'now')); ?></span>
          </div>
          <p class="desc"><?php echo h($a['short'] ?? ''); ?></p>
          <div class="actions">
            <a class="btn" href="<?php echo h($pretty); ?>">Read Full Story</a>
            <button class="btn ghost copy-btn" data-link="<?php echo h($abs); ?>">Copy Link</button>
          </div>
        </div>
      </article>
    <?php endforeach; ?>

    <?php if ($pages>1): ?>
      <nav class="pager" aria-label="Pagination">
        <?php
          $link=function($page) use ($tagQueryRaw){
            $p=[]; if($tagQueryRaw!=='') $p['tag']=$tagQueryRaw; if($page>1) $p['pg']=$page;
            $qs2=$p?('?'.http_build_query($p)) : '';
            return 'index.php'.$qs2;
          };
          $range=[]; $range[1]=true; $range[$pages]=true;
          for($i=max(1,$pg-2); $i<=min($pages,$pg+2); $i++) $range[$i]=true; $last=0;
        ?>
        <?php if ($pg>1): ?><a href="<?php echo h($link($pg-1)); ?>">« Prev</a><?php endif; ?>
        <?php for($i=1;$i<=$pages;$i++): if(!isset($range[$i])) continue; ?>
          <?php if($last && $i>$last+1) echo '<span>…</span>'; ?>
          <?php echo ($i==$pg)? '<span class="current">'.$i.'</span>' : '<a href="'.h($link($i)).'">'.$i.'</a>'; ?>
          <?php $last=$i; ?>
        <?php endfor; ?>
        <?php if ($pg<$pages): ?><a href="<?php echo h($link($pg+1)); ?>">Next »</a><?php endif; ?>
      </nav>
    <?php endif; ?>
  </section>
</main>

<footer class="site-footer"><div>Crypto News © <?php echo date('Y'); ?> • All rights reserved</div></footer>
<script src="live-ticker.js?v=<?php echo filemtime('live-ticker.js'); ?>"></script>
</body>
</html>